/* vim: set ts=8 sw=4 tw=0 noet : */
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/mutex.h>
#include <linux/rwsem.h>
#include <linux/list.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/fs.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Sergej Bauer");

#define DIRNAME "multipipe"
#define WRITENAME "write"
#define READNAME "read"

static struct proc_dir_entry *dir_entry;

static struct kmem_cache *mp_reader_cache = NULL;
static struct kmem_cache *mp_flow_cache = NULL;

struct mp_flow_t {
	char data;
	struct list_head list;
};

struct mp_reader_t {
	char *buf;
	int nbytes;
	wait_queue_head_t wq;
	struct mutex lock;
	struct list_head flow;
	struct list_head list;
};

#define LIMIT	(8 * 1024 * 1024)

static atomic_t nreaders = ATOMIC_INIT(0), nbtotal = ATOMIC_INIT(0),
	d_invoked = ATOMIC_INIT(0);
static DECLARE_WAIT_QUEUE_HEAD(wq);
static LIST_HEAD(readers);
static struct mutex readers_sem;
static struct mp_reader_t *dummy;

ssize_t mp_read(struct file *filp, char __user *buffer, size_t count,
		loff_t *offset)
{
	struct mp_reader_t *r = filp->private_data;
	struct mp_flow_t *f;
	int ret, n = 0, flag = 0;

	if (count == 0) return 0;

	ret = wait_event_interruptible(r->wq, !list_empty(&r->flow));
	if (ret < 0)
		return ret;

	count = min(count, (size_t)PAGE_SIZE);

	if (r->buf == NULL) {
		r->buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
		if (!r->buf)
			return -ENOMEM;
	}

	mutex_lock(&r->lock);
	do {
		f = list_entry(r->flow.next, struct mp_flow_t, list);
		r->buf[r->nbytes++] = f->data;
		n++;
		if (list_is_last(r->flow.next, &r->flow) || (r->nbytes >= count))
			flag = 1;

		list_del(r->flow.next);
		kmem_cache_free(mp_flow_cache, f);
		atomic_dec(&nbtotal);
	} while (!flag);
	mutex_unlock(&r->lock);

	if (copy_to_user(buffer, r->buf + r->nbytes - n, n)) {
		r->nbytes -= n;
		return -EFAULT;
	}

	kfree(r->buf);
	r->buf = NULL;
	r->nbytes = 0;
	if (atomic_read(&nbtotal) == 0)
		wake_up_interruptible(&wq);

	*offset += n;
	return n;
}

ssize_t mp_write(struct file *filp, const char __user *buffer, size_t count,
		 loff_t *offset)
{
	struct mp_reader_t *r;
	struct mp_flow_t *f;
	struct list_head *ir, *t;
	char *buf;
	int i, ret;

	ret = wait_event_interruptible(wq, (atomic_read(&nbtotal) < LIMIT));
	if (ret < 0)
		return ret;

	count = min(count, (size_t)PAGE_SIZE);
	buf = kmalloc(count, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;
	if (copy_from_user(buf, buffer, count)) {
		kfree(buf);
		return -EFAULT;
	}

	mutex_lock(&readers_sem);
	list_for_each_safe(ir, t, &readers) {
		r = list_entry(ir, struct mp_reader_t, list);
		if ((r == dummy) && (atomic_read(&nreaders) > 1))
			continue;
		mutex_unlock(&readers_sem);

		mutex_lock(&r->lock);
		for (i = 0; i < count; i++) {
			f = kmem_cache_alloc(mp_flow_cache, GFP_KERNEL);
			if (!f) {
				kfree(buf);
				mutex_unlock(&r->lock);
				return -ENOMEM;
			}
			f->data = buf[i];
			list_add_tail(&f->list, &r->flow);
			if (atomic_read(&d_invoked) &&
					atomic_read(&nbtotal) > 0) {
				f = list_entry(r->flow.next, struct mp_flow_t,
					       list);
				list_del(r->flow.next);
				kmem_cache_free(mp_flow_cache, f);
			}
			else
				atomic_inc(&nbtotal);
		}
		mutex_unlock(&r->lock);
		wake_up_interruptible(&r->wq);
		mutex_lock(&readers_sem);
	}
	mutex_unlock(&readers_sem);

	kfree(buf);

	*offset += count;
	return count;
}

loff_t mp_lseek(struct file *filp, loff_t offset, int whence)
{
	struct mp_reader_t *r = filp->private_data;
	struct mp_flow_t *f;
	int ret, n = 0;

	if ((offset < 0) || (whence != SEEK_CUR))
		return -ESPIPE;

	while (n < offset) {
		ret = wait_event_interruptible(r->wq, !list_empty(&r->flow));
		if (ret < 0)
			return ret;

		mutex_lock(&r->lock);
		f = list_entry(r->flow.next, struct mp_flow_t, list);
		list_del(r->flow.next);
		mutex_unlock(&r->lock);

		n++;
		kmem_cache_free(mp_flow_cache, f);
		atomic_dec(&nbtotal);
	}

	return offset;
}

int mp_open(struct inode *inode, struct file *filp)
{
	struct mp_reader_t *pd;
	struct mp_flow_t *flow, *t;
	struct list_head *it;
	int f = 0, nr = 0;

	nr = atomic_inc_return(&nreaders);
	pd = kmem_cache_alloc(mp_reader_cache, GFP_KERNEL);
	if (!pd) {
		atomic_dec(&nreaders);
		return -ENOMEM;
	}

	mutex_init(&pd->lock);
	init_waitqueue_head(&pd->wq);
	INIT_LIST_HEAD(&pd->flow);
	pd->nbytes = 0;
	pd->buf = NULL;

	mutex_lock(&dummy->lock);
	if (atomic_read(&d_invoked)) {
		list_for_each(it, &dummy->flow) {
			flow = list_entry(it, struct mp_flow_t, list);
			t = kmem_cache_alloc(mp_flow_cache, GFP_KERNEL);
			if (!t) {
				mutex_unlock(&dummy->lock);
				kmem_cache_free(mp_reader_cache, pd);
				atomic_dec(&nreaders);
				return -ENOMEM;
			}
			t->data = flow->data;
			list_add_tail(&t->list, &pd->flow);
		}

		mutex_lock(&readers_sem);
		if ((nr > 0) && (atomic_read(&nreaders) <= nr)) {
			list_del(&dummy->list);
			while (!list_empty(&dummy->flow)) {
				flow = list_entry(dummy->flow.next,
						  struct mp_flow_t, list);
				list_del(dummy->flow.next);
				kmem_cache_free(mp_flow_cache, flow);
			}
			atomic_set(&d_invoked, 0);
		}
		mutex_unlock(&readers_sem);
		f = 1;
	}
	mutex_unlock(&dummy->lock);

	filp->private_data = pd;
	mutex_lock(&readers_sem);
	list_add_tail(&pd->list, &readers);
	mutex_unlock(&readers_sem);
	if (f) wake_up_interruptible(&pd->wq);

	return 0;
}

int mp_release(struct inode *inode, struct file *filp)
{
	struct mp_reader_t *pd = filp->private_data;
	struct mp_flow_t *f;

	mutex_lock(&readers_sem);
	mutex_lock(&pd->lock);
	if (pd->buf) kfree(pd->buf);
	while (!list_empty(&pd->flow)) {
		f = list_entry(pd->flow.next, struct mp_flow_t, list);
		list_del(pd->flow.next);
		kmem_cache_free(mp_flow_cache, f);
	}
	list_del(&pd->list);
	mutex_unlock(&pd->lock);
	mutex_unlock(&readers_sem);
	kmem_cache_free(mp_reader_cache, pd);

	if (atomic_dec_return(&nreaders) == 0) {
		mutex_lock(&dummy->lock);
		while (!list_empty(&dummy->flow))
			list_del(dummy->flow.next);
		mutex_unlock(&dummy->lock);

		mutex_lock(&readers_sem);
		list_add(&dummy->list, &readers);
		mutex_unlock(&readers_sem);
		atomic_set(&d_invoked, 1);
	}

	mutex_destroy(&pd->lock);
	mutex_destroy(&dummy->lock);
	mutex_destroy(&readers_sem);

	return 0;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(5, 6, 0)
static struct file_operations rqp_fops = {
	.owner		= THIS_MODULE,
	.read		= mp_read,
	.open		= mp_open,
	.llseek		= mp_lseek,
	.release	= mp_release,
};
static struct file_operations wqp_fops = {
	.owner		= THIS_MODULE,
	.write		= mp_write,
};
#else
static const struct proc_ops rqp_fops = {
	.proc_read		= mp_read,
	.proc_open		= mp_open,
	.proc_lseek		= mp_lseek,
	.proc_release		= mp_release,
};
static const struct proc_ops wqp_fops = {
	.proc_write		= mp_write,
};
#endif

static int __init init_mp(void)
{
	struct proc_dir_entry *write_entry, *read_entry;

#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)
	dir_entry = proc_mkdir(DIRNAME, NULL);
	read_entry = create_proc_entry(READNAME, 0444, dir_entry);
	write_entry = create_proc_entry(WRITENAME, 0222, dir_entry);
#else
	dir_entry = proc_mkdir(DIRNAME, NULL);
	read_entry = proc_create(READNAME, 0444, dir_entry, &rqp_fops);
	write_entry = proc_create(WRITENAME, 0222, dir_entry, &wqp_fops);
#endif
	if (!dir_entry || !write_entry || !read_entry)
		return -EPERM;
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)
	read_entry->proc_fops = &rqp_fops;
	write_entry->proc_fops = &wqp_fops;
#endif
	mp_reader_cache = kmem_cache_create("mp_reader_cache",
					    sizeof(struct mp_reader_t),
					    0, 0, 0);
	if (!mp_reader_cache)
		goto reader_cache_err;

	mp_flow_cache = kmem_cache_create("mp_flow_cache",
					  sizeof(struct mp_flow_t),
					  0, 0, 0);
	if (!mp_flow_cache)
		goto flow_cache_err;

	dummy = kmem_cache_alloc(mp_reader_cache, GFP_KERNEL);
	if (!dummy)
		goto dummy_err;

	mutex_init(&readers_sem);

	mutex_init(&dummy->lock);
	init_waitqueue_head(&dummy->wq);
	INIT_LIST_HEAD(&dummy->flow);
	dummy->buf = NULL;
	dummy->nbytes = 0;
	mutex_lock(&readers_sem);
	list_add(&dummy->list, &readers);
	mutex_unlock(&readers_sem);
	atomic_set(&d_invoked, 1);

	return 0;

dummy_err:
	kmem_cache_destroy(mp_flow_cache);

flow_cache_err:
	kmem_cache_destroy(mp_reader_cache);

reader_cache_err:
	remove_proc_entry(WRITENAME, dir_entry);
	remove_proc_entry(READNAME, dir_entry);
	remove_proc_entry(DIRNAME, NULL);

	return -EFAULT;
}

static void __exit exit_mp(void)
{
	struct mp_reader_t *r;
	struct mp_flow_t *f;

	remove_proc_entry(WRITENAME, dir_entry);
	remove_proc_entry(READNAME, dir_entry);
	remove_proc_entry(DIRNAME, NULL);

	mutex_lock(&readers_sem);
	while (!list_empty(&readers)) {
		r = list_entry(readers.next, struct mp_reader_t, list);

		mutex_lock(&r->lock);
		while (!list_empty(&r->flow)) {
			f = list_entry(r->flow.next, struct mp_flow_t, list);
			list_del(r->flow.next);
			kmem_cache_free(mp_flow_cache, f);
		}
		mutex_unlock(&r->lock);
		list_del(readers.next);
		kmem_cache_free(mp_reader_cache, r);
	}
	mutex_unlock(&readers_sem);

	kmem_cache_destroy(mp_flow_cache);
	kmem_cache_destroy(mp_reader_cache);
}

module_init(init_mp);
module_exit(exit_mp);

