#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>

#define T_READER 7
#define T_WRITER 11

#define MULTIPIPE

#define FILE_TO_READ "/proc/multipipe/read"
#define FILE_TO_WRITE "/proc/multipipe/write"

#define BSIZE 555

#define MAX_THREADS 8

#define ERR "CHECK ERROR: "


unsigned char buf[MAX_THREADS][BSIZE];
unsigned char sum[MAX_THREADS];
unsigned int ops[MAX_THREADS];
unsigned long long bytes[MAX_THREADS];
int fd[MAX_THREADS];

#define TERMINATED ((pthread_t)-3)

volatile char flags[MAX_THREADS];
pthread_t threads[MAX_THREADS];

volatile int intr;

#define GRACE (2 + (MAX_THREADS/8)*2)


static inline unsigned char chk(unsigned char *b, int count)
{
	unsigned char * end = b + count, c = 0;
	while (b<end)
		c += *b++;
	return c;
}

static void rnd(unsigned char *b, int count)
{
	unsigned char * end = b + count;
	while (b<end)
		*b++ = random()>>5;
}

void *thread(void*arg)
{
	int i = (long int)arg, res, ireader = flags[i] == T_READER;
	unsigned char *b = buf[i];
	
	rnd(b, BSIZE);
	
	while(intr <= ireader) {
		int sz = random()%BSIZE;
		
		if (ireader) {
			res = read(fd[i], b, sz);
			if (res<0) {
				perror(ERR "read");
				continue;
			}
			if (!res && sz)
				fprintf(stderr, ERR "reader: EOF??\n");
			
			if (res > sz)
				fprintf(stderr, ERR "reader: read  %d more than asked! %d\n", res, sz);
		} else {
			
			if ((sz & 30) == 30)
				rnd(b, sz);				
			 
			res = write(fd[i], b, sz);
			if (res<0) {
				perror("write");
				continue;
			}
			
			if (res != sz)
				fprintf(stderr, ERR "writer: I sent %d, was sent %d\n", sz, res);
		}
		
		sum[i] += chk(b, res);
		bytes[i] += res;
		ops[i]++;
		
		if (!(random()&7))
			sched_yield();
	}
	
	threads[i] = TERMINATED;
	
	fprintf(stderr, "exiting %d[%c]\n", i, ireader ? 'r' : 'w');
	return NULL;
}

int num_readers, num_writers;

void print_stats()
{
	unsigned long long rb = 0, wb = 0;
	unsigned int rops = 0, wops = 0;
	unsigned rs = 0, ws = 0;
	
	int i;
	
	for (i=0; i<MAX_THREADS; i++)
		if (flags[i] == T_READER) {
			rb += bytes[i];
			rops += ops[i];
			rs += sum[i];	
		} else {
			wb += bytes[i];
			wops += ops[i];
			ws += sum[i];	
		}
	
	fprintf(stderr, "ops r:%u, w:%u | bytes r:%llu w:%llu | sum r:%x, w:%x \n",
		rops, wops, rb, wb, rs&255, ws&255);
		
#ifdef MULTIPIPE
	for (i=0; i<MAX_THREADS; i++)
		if (flags[i] == T_READER)
			fprintf(stderr, " %lld(%02x) ", wb - bytes[i], (ws-sum[i]) & 255);
	fprintf(stderr, "\n\n");
#endif
}


void sig(int n)
{
	fprintf(stderr, "Caught signal #%d, terminating, grace period %d seconds ...\n", n, GRACE);
	intr = 1;
}

int main(int argc, char **argv)
{
	int i, nr = 0;
	void * res;
	
	srandom((int)time(NULL));

	if (argc == 1) {	
		for (i=0; i<MAX_THREADS; i++) {
			if (random() & 8 /*i == 0*/) {
				flags[i] = T_READER;
				num_readers++;
				fd[i] = open(FILE_TO_READ, O_RDONLY, 0);
			} else {
				flags[i] = T_WRITER;
				num_writers++;
				fd[i] = open(FILE_TO_WRITE, O_WRONLY, 0);
			}
			if (fd[i] < 0) {
				perror("opening file");	
				return 1;
			}	
		}
	} else {
		nr = atoi(argv[1]);
		for (i=0; i<MAX_THREADS; i++) {
			if (i < nr) {
				flags[i] = T_READER;
				num_readers++;
				fd[i] = open(FILE_TO_READ, O_RDONLY, 0);
			} else {
				flags[i] = T_WRITER;
				num_writers++;
				fd[i] = open(FILE_TO_WRITE, O_WRONLY, 0);
			}
			if (fd[i] < 0) {
				perror("opening file");	
				return 1;
			}	
		}
	}

	for (i=0; i<MAX_THREADS; i++) {
		if (pthread_create(threads+i, NULL, thread, (void *)((unsigned long long)i)))
				abort();
		fprintf(stderr, ".");	
	}
	
	fprintf(stderr, " started, %d readers, %d writers\n", num_readers, num_writers);
	
	signal(SIGINT, sig);
	signal(SIGTERM, sig);
	
	while (!intr) {
		print_stats();
		sleep(10);	
	}

	sleep(GRACE/2);
	fprintf(stderr, "\n");
	intr = 2;
	sleep(GRACE/2);
	fprintf(stderr, "\n");


	for (i=0; i<MAX_THREADS; i++)
		if (threads[i] != TERMINATED) {
			fprintf(stderr, "canceling %d[%c] ...", i, flags[i] == T_READER ? 'r' : 'w');		
			pthread_cancel(threads[i]);
			fprintf(stderr, " + \n");
			pthread_join(threads[i], &res);
		}

	fprintf(stderr, "\n");
	print_stats();
	return 0;
}
